﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using BusinessHelper = VeteransAffairs.Registries.Business.Utilities.Helpers;

public partial class TBI_Instruments_TimedUpAndGo : BaseTBIInstrumentPage {
    #region Variable Declarations

    PatientManager patientManager = new PatientManager();
    InstrumentType instrumentType = new InstrumentType();

    protected int _surveyTypeId = 22; // Database Id for Timed Up and Go

    SURVEY _survey;   // the root object in the Survey set
    SURVEY_RESULT _surveyResult; // the object representing the actual data in the Survey

    List<STD_QUESTION> QuestionList = new List<STD_QUESTION>();

    #endregion

    /// <summary>
    /// Default Constructor
    /// </summary>
    public TBI_Instruments_TimedUpAndGo() {
        //Manager Save Notification Event Handler
        patientManager.BOSaveSuccess += manager_BOSaveSuccess;
    }

    #region Protected Methods

    /// <summary>
    /// Page Load method
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e) {
        TBIServiceTO _tbiServiceTo;

        try {
            _tbiServiceTo = (TBIServiceTO)Session["TBIServiceTO"];
            _tbiInstrumentManager.TbiServiceTo = _tbiServiceTo;

            string instTitle = _tbiInstrumentManager.GetInstrumentByID(_surveyTypeId).Name;

            LabelBreadCrumb.Text = "TBI Instruments > Instrument Associations > " + instTitle;
            LabelInstrumentTitle.Text = instTitle;
            Page.Title = "TBI Instruments - " + instTitle;

            if (_tbiInstrumentManager.TbiServiceTo != null) {
                Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
                labelUser.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.UserName);

                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
        } catch (Exception ex) {
            //Redirect to message that session is ended and need to log back in from CPRS.
            Exception _tbilogerror = HandleTBIException(ex, "TimedUpAndGo.aspx", "100027", "SESSIOENDED");
            Response.Redirect("Warning.aspx?err=2", false);
        }

        if (!IsPostBack) {
            if (instrumentType.Status != null) {
                _tbiInstrumentManager.TbiServiceTo.SelectedInstrumentType = instrumentType;
            }

            string _snum = "";
            if (_tbiInstrumentManager.TbiServiceTo != null) {
                _snum = _tbiInstrumentManager.TbiServiceTo.PatientSnum;
            }

            if (!string.IsNullOrEmpty(_snum))
            {
                string patICN = patientManager.GetPatientICNBySnum(_snum);

                if (!string.IsNullOrEmpty(patICN)) {
                    ViewState["PatientICN"] = patICN;
                    RefreshSurvey(patICN);
                } else {
                    throw new ArgumentException("Patient with this SSN does not exist.");
                }

                if (_survey != null && !_survey.isNew) {
                    SetUpSurvey(_survey);
                    //calculateTotal();
                    //AssistiveDevice_SelectedIndexChanged();
                }
                string referrer = "";
                if (Request.UrlReferrer != null) {
                    referrer = Request.UrlReferrer.PathAndQuery;
                }
                ViewState["referrer"] = referrer;
            }

            //Check if the request is to start a new item 
            //a new item page request is signified when the querystring parameter id equals zero or is not present
            if (_tbiInstrumentManager.TbiServiceTo == null)
            {
                Logging.WriteLogToFile("TimedUpAndGo.Page_Load(): WarningRedirect: _tbiInstrumentManager.TbiServiceTo == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            } else {
                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum.Trim().Length == 9 ? String.Format("{0:000-00-0000}", int.Parse(_tbiInstrumentManager.TbiServiceTo.PatientSnum)) : _tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
        }
        QuestionList = GetAllQuestionsForSurvey();

        // Set control values for Diagnosis Question
        ctrlDiagnosis.QuestionText = GetQuestionText("DX");
        ctrlDiagnosis.SurveyTypeCode = HiddenSurveyType.Value;

        LabelDate1.Text = GetQuestionText("D");
        LabelForQ1.Text = GetQuestionText("1");
        LabelForQ2.Text = GetQuestionText("2");
        LabelForQ3.Text = GetQuestionText("3");
        LabelForQ4.Text = GetQuestionText("4");
        LabelForQ5.Text = GetQuestionText("5");
        LabelForQ6.Text = GetQuestionText("6");
        LabelForQ7.Text = GetQuestionText("7");
        dpDate1.MaxDate = DateTime.Now;
    }


    protected override void LoadViewState(object savedState) {
        base.LoadViewState(savedState);
        string surveyXml = (string)ViewState["surveyXml"];
        _survey = SerializerUtil.Deserialize<SURVEY>(surveyXml);
    }

    protected override object SaveViewState() {
        ViewState["surveyXml"] = _survey.Serialize();
        return base.SaveViewState();
    }

    /// <summary>
    /// Returns the Question_Text of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected string GetQuestionText(string questionId) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionId.ToLower()) {
                return ques.QUESTION_TEXT;
            }
        }

        return "";
    }

    /// <summary>
    /// Returns the Question_Id of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected int GetQuestionId(string questionNumber) {
        foreach (STD_QUESTION ques in QuestionList) {
            if (ques.QUESTION_NUMBER.ToLower() == questionNumber.ToLower()) {
                return ques.ID;
            }
        }

        return 0;
    }

    /// <summary>
    /// Save Draft button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    public void btnSaveSurveyDraft_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        SetUpSurvey(_survey);
    }

    /// <summary>
    /// Save and Prepare Note button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnSubmit_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSubmitSurvey.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);

        if (status > 0) {
            Session["survey"] = _survey;
            Response.Redirect("InstrumentSummaryNote.aspx", false);
        } else {
            SetUpSurvey(_survey); //Follow Save Draft logic 
        }
    }

    /// <summary>
    /// Inactive timer firing.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void InActiveTimer_Tick(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";

        Page.Validate();

        if (Page.IsValid) {
            SaveCompleteSurvey(ref status);
            //SetUpSurvey(_survey);
            Response.Redirect("TimedUpAndGo.aspx", false);
        } else {
            SaveFailMessage = "Auto-Save Validation Failed. Please complete required fields in the survey and press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        }
    }

    /// <summary>
    /// Cancel button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnCancel_Click(object sender, EventArgs e) {
        ViewState["surveyXml"] = null;
        ViewState["PatientICN"] = null;
        ViewState["referrer"] = null;
        Session.Remove("survey");
        Response.Redirect("Instrument.aspx", false);
        //this.ClientScript.RegisterStartupScript(typeof(Page), "closePage", "<script type='text/javascript'> window.open('','_self','');window.close(); </script>");
    }

    /// <summary>
    /// Hide or Show instructions
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void lnkBtnToggleInstructions_OnClick(object sender, EventArgs e) {
        if (PanelInstructions.Visible) {
            PanelInstructions.Visible = false;
            lnkBtnToggleInstructions.Text = "Click for Instructions";
        } else {
            PanelInstructions.Visible = true;
            lnkBtnToggleInstructions.Text = "Hide Instructions";
        }
    }

    /// <summary>
    /// Fired when the Assistive Device Selection changes.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void rblQ1_SelectedIndexChanged(object sender, EventArgs e) {
        if (rblQ1.SelectedItem.Text.ToLower() == "yes") {
            pnlAssistiveDeviceSelection.Visible = true;
        } else {
            pnlAssistiveDeviceSelection.Visible = false;
            rblQ2.SelectedIndex = -1;
        }
    }

    #endregion


    #region Private Methods

    /// <summary>
    /// Get all questions for this survey
    /// </summary>
    /// <returns></returns>
    private List<STD_QUESTION> GetAllQuestionsForSurvey() {
        return _tbiInstrumentManager.SelectAllQuestions(_surveyTypeId).ToList();
    }

    /// <summary>
    /// Refresh Survey
    /// </summary>
    /// <param name="patientICN"></param>
    private void RefreshSurvey(string patientICN) {
        var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

        if (surveys.Count > 0) {
            _survey = surveys.First();
        } else {
            _survey = new SURVEY();
        }

        _survey.SetAsChangeTrackingRoot();
    }

    /// <summary>
    /// Save Complete Survey
    /// </summary>
    /// <param name="status"></param>
    private void SaveCompleteSurvey(ref Int32 status) {
        Page.Validate();
        if ((Page.IsValid)) {
            string patientICN = String.Empty;

            status = 0;

            //Retrieve patient's ICN from the session variable
            if (ViewState["PatientICN"] != null) {
                patientICN = ViewState["PatientICN"].ToString();
                if (String.IsNullOrEmpty(patientICN.Trim()))
                {
                    Logging.WriteLogToFile("TimedUpAndGo.SaveCompletedSurvey(): WarningRedirect: String.IsNullOrEmpty(patientICN.Trim()");
                    //Lost state. Redirect to message that need to log back in from CPRS.
                    Response.Redirect("Warning.aspx?err=2", false);
                }
            }
            else
            {
                Logging.WriteLogToFile("TimedUpAndGo.SaveCompletedSurvey(): WarningRedirect: ViewState[PatientICN] != null");
                //Lost state. Redirect to message that need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }

            if (_survey != null) {
                if (_survey.isNew) {
                    //Ensure that a concurrent session hasn't already inserted a record for this survey
                    var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

                    if (surveys.Count > 0)
                    {
                        Logging.WriteLogToFile("TimedUpAndGo.SaveCompletedSurvey(): WarningRedirect: surveys.Count > 0");
                        //A concurrent session has already inserted a record for this survey and it is not new anymore.
                        Response.Redirect("Warning.aspx?err=9", false);
                    } else {
                        _survey.SURVEY_STATUS = "inprocess";

                        _survey.SetAsInsertOnSubmit();

                        _survey.patient_ICN = patientICN;

                        _survey.STD_SURVEY_TYPE_ID = _surveyTypeId;

                        // TODO: change to datepicker
                        _survey.SURVEY_DATE = DateTime.Today;

                        //Save survey if it's new before adding any responses
                        SuppressSaveMessage = true;
                        _tbiInstrumentManager.Update(_survey);
                        SuppressSaveMessage = false;

                        RefreshSurvey(patientICN);
                    }
                }

                SaveSurvey(ref _survey);

                _survey.SetAsUpdateOnSubmit();
                status = _tbiInstrumentManager.Update(_survey);

                //Fortify errors on this line (this.Master as RegistryInstrumentMaster).SetDirtyFlag(false);
                ((RegistryInstrumentMaster)Master).SetDirtyFlag(false);

                RefreshSurvey(patientICN);
                SuppressSaveMessage = false;
            }
        }
    }


    /// <summary>
    /// Save survey.
    /// </summary>
    /// <param name="survey"></param>
    private void SaveSurvey(ref SURVEY survey) {
        List<STD_QUESTION> questions = GetAllQuestionsForSurvey();

        //Delete all the survey results from database if there are any.
        SURVEY surveyWithResults = _tbiInstrumentManager.GetSurveyWithAllAnswers(survey.SURVEYS_ID);
        _tbiInstrumentManager.DeleteAllSurveyResults(surveyWithResults);


        // Question DX
        _surveyResult = new SURVEY_RESULT();
        try {
            // Add Question DX choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(ctrlDiagnosis.GetSelectedId());
            _surveyResult.RESULT_TEXT = ctrlDiagnosis.GetSelectedResultText();
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question DX");
        }


        // Date
        _surveyResult = new SURVEY_RESULT();
        if (dpDate1.SelectedDate != null) {
            STD_QUESTION questionD = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "D");
            _surveyResult = new SURVEY_RESULT();
            try {
                // Add Question DTE choice
                _surveyResult.STD_QUESTION_CHOICE_ID = questionD.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = ((DateTime)dpDate1.SelectedDate).ToShortDateString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question D");
            }
        }


        // Question 1 - Assistive device
        _surveyResult = new SURVEY_RESULT();
        try {
            // Add Question 1 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQ1.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQ1.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 1");
        }

        // Question 2 - Assistive device selection
        _surveyResult = new SURVEY_RESULT();
        try {
            // Add Question 2 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQ2.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQ2.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 2");
        }

        // Question 3 - Time
        STD_QUESTION question3 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "3");
        _surveyResult = new SURVEY_RESULT();
        try {
            _surveyResult.STD_QUESTION_CHOICE_ID = question3.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = txtQ3.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 3");
        }


        // Question 4 - Distance
        STD_QUESTION question4 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "4");
        _surveyResult = new SURVEY_RESULT();
        try {
            _surveyResult.STD_QUESTION_CHOICE_ID = question4.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = txtQ4.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 4");
        }


        // Question 5 - Time - Test 2
        STD_QUESTION question5 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "5");
        _surveyResult = new SURVEY_RESULT();
        try {
            _surveyResult.STD_QUESTION_CHOICE_ID = question5.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = txtQ5.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 5");
        }


        // Question 6 - Distance - Test 2
        STD_QUESTION question6 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "6");
        _surveyResult = new SURVEY_RESULT();
        try {
            _surveyResult.STD_QUESTION_CHOICE_ID = question6.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            _surveyResult.RESULT_TEXT = txtQ6.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 6");
        }

        // Question 7 - Unstable on turning
        _surveyResult = new SURVEY_RESULT();
        try {
            // Add Question 7 choice
            _surveyResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(rblQ7.SelectedValue);
            _surveyResult.RESULT_TEXT = rblQ7.SelectedItem.Text;
            survey.SURVEY_RESULTs.Add(_surveyResult);
        } catch (Exception surveyEx) {
            Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question 7");
        }

        // determine if multiple tests were done
        if (!String.IsNullOrEmpty(txtQ5.Text) && !String.IsNullOrEmpty(txtQ6.Text)) {
            int time1 = Convert.ToInt32(txtQ3.Text);
            int distance1 = Convert.ToInt32(txtQ4.Text);

            int time2 = Convert.ToInt32(txtQ5.Text);
            int distance2 = Convert.ToInt32(txtQ6.Text);

            decimal averageTime = ((time1 + time2) / 2);
            decimal averageDist = ((distance1 + distance2) / 2);

            HiddenAverageTime.Value = averageTime.ToString();
            HiddenAverageDistance.Value = averageDist.ToString();

            // Question 8 - Average Time
            STD_QUESTION question8 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "8");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 8 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question8.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = averageTime.ToString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question8");
            }

            // Question 9 - Average Distance
            STD_QUESTION question9 = questions.FirstOrDefault(Q => Q.QUESTION_NUMBER == "9");
            _surveyResult = new SURVEY_RESULT();

            try {
                // Add Question 9 choice
                _surveyResult.STD_QUESTION_CHOICE_ID = question9.STD_QUESTION_CHOICEs.FirstOrDefault().STD_QUESTION_CHOICE_ID;
                _surveyResult.RESULT_TEXT = averageDist.ToString();
                survey.SURVEY_RESULTs.Add(_surveyResult);
            } catch (Exception surveyEx) {
                Exception _tbilogerror = HandleTBIException(surveyEx, "TimedUpAndGo.aspx", "100027", "SaveSurvey() - Question9");
            }
        }

    }// END SaveSurvey


    /// <summary>
    /// Initialize/setup controls
    /// </summary>
    /// <param name="survey"></param>
    private void SetUpSurvey(SURVEY survey) {
        try {
            foreach (var sResult in survey.SURVEY_RESULTs) {
                switch (sResult.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER) {
                    case "DX":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            ctrlDiagnosis.SelectedChoiceId = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();

                            if (sResult.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other")) {
                                ctrlDiagnosis.SetOtherDescriptionText(sResult.RESULT_TEXT);
                            }
                        } else {
                            ctrlDiagnosis.SelectedChoiceId = null;
                        }
                        break;


                    case "D":
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            DateTime temp;
                            if (DateTime.TryParse(sResult.RESULT_TEXT, out temp)) {
                                dpDate1.SelectedDate = temp;
                            }
                        }
                        break;

                    case "1":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQ1.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQ1.SelectedIndex = -1;
                        }
                        break;

                    case "2":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQ2.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQ2.SelectedIndex = -1;
                        }
                        break;
                    case "3":
                        txtQ3.Text = sResult.RESULT_TEXT;
                        break;
                    case "4":
                        txtQ4.Text = sResult.RESULT_TEXT;
                        break;
                    case "5":
                        txtQ5.Text = sResult.RESULT_TEXT;
                        break;
                    case "6":
                        txtQ6.Text = sResult.RESULT_TEXT;
                        break;

                    case "7":
                        if (sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1) {
                            rblQ7.SelectedValue = sResult.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
                        } else {
                            rblQ7.SelectedIndex = -1;
                        }
                        break;

                    case "8":
                        // Average Time:
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            HiddenAverageTime.Value = sResult.RESULT_TEXT;
                        }
                        break;
                    case "9":
                        // Average Distance
                        if (!String.IsNullOrEmpty(sResult.RESULT_TEXT)) {
                            HiddenAverageDistance.Value = sResult.RESULT_TEXT;
                        }
                        break;


                    default:
                        break;
                }
            }
        } catch (Exception ex) {
            Exception _tbilogerror = HandleTBIException(ex, "TimedUpAndGo.aspx", "100027", "SetUpSurvey()");
        }
    }
    // END SetUpSurvey

    #endregion
}
